<?php

namespace Tests\Browser\Modules\Product;

use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\File;
use Laravel\Dusk\Browser;
use Modules\Product\Entities\Brand;
use Modules\Product\Entities\Category;
use Modules\Product\Entities\Product;
use Modules\Product\Entities\ProductCrossSale;
use Modules\Product\Entities\ProductRelatedSale;
use Modules\Product\Entities\ProductTag;
use Modules\Product\Entities\ProductUpSale;
use Modules\Product\Entities\ProductVariations;
use Modules\Product\Entities\UnitType;
use Modules\Seller\Entities\SellerProduct;
use Modules\Shipping\Entities\ProductShipping;
use Modules\Shipping\Entities\ShippingMethod;
use Tests\DuskTestCase;

class InHouseProductTest extends DuskTestCase
{
    use WithFaker;

    protected $categories =[];
    protected $brands =[];
    protected $units =[];
    protected $shipping_methods =[];

    public function setUp(): void
    {
        parent::setUp();

        for ($i = 0; $i < 3; $i++) {
            $this->categories[] = Category::create([
                'name' => $this->faker->name,
                'slug' => $this->faker->slug,
                'parent_id' => 0
            ]);


            $this->brands[] = Brand::create([
                'name' => $this->faker->name,
                'slug' => $this->faker->slug,
                'status' => 1,
                'logo' => 'uploads/images/29-05-2021/60b1e99781fbb.png'
            ]);

            $this->units[] = UnitType::create([
                'name' => $this->faker->name,
                'status' => 1
            ]);

            $this->shipping_methods[] = ShippingMethod::create([
                'method_name' => $this->faker->name,
                'phone' => $this->faker->phoneNumber,
                'shipping_time' => '8-12 days',
                'cost' => 5,
                'is_active' => 1
            ]);
        }

    }

    public function tearDown(): void
    {
        foreach ($this->categories as $category) {
            $category->delete();
        }
        foreach ($this->brands as $brand) {
            $brand->delete();
        }
        foreach ($this->units as $unit) {
            $unit->delete();
        }
        foreach ($this->shipping_methods as $shipping_method) {
            $shipping_method->delete();
        }

        $products = Product::all();
        foreach($products as $product){
            
            ProductVariations::destroy($product->variations->pluck('id'));
            foreach($product->skus as $sku){
                if(File::exists(public_path($sku->variant_image))){
                    File::delete(public_path($sku->variant_image));
                }
                if(isset($sku->digital_file)){
                    if($sku->digital_file->file_source != null){
                        if(File::exists(public_path($sku->digital_file->file_source))){
                            File::delete(public_path($sku->digital_file->file_source));
                        }
                    }
                    $sku->digital_file->delete();
                }
                $sku->delete();
            }
            ProductTag::destroy($product->tags->pluck('id'));

            foreach($product->gallary_images as $image){
                if(File::exists(public_path($image->images_source))){
                    File::delete(public_path($image->images_source));
                }
                
                $image->delete();
            }
            if(File::exists(public_path($product->thumbnail_image_source))){
                File::delete(public_path($product->thumbnail_image_source));
            }

            if($product->meta_image != null){
                if(File::exists(public_path($product->meta_image))){
                    File::delete(public_path($product->meta_image));
                }
            }

            if($product->pdf != null){
                if(File::exists(public_path($product->pdf))){
                    File::delete(public_path($product->pdf));
                }
            }
            

            ProductRelatedSale::destroy($product->relatedProducts->pluck('id'));
            ProductUpSale::destroy($product->upSales->pluck('id'));
            ProductCrossSale::destroy($product->crossSales->pluck('id'));
            ProductShipping::destroy($product->shippingMethods->pluck('id'));
            $product->delete();
        }

        parent::tearDown(); // TODO: Change the autogenerated stub
    }

    /**
     * A Dusk test example.
     *
     * @return void
     */
    public function test_for_visit_index_page(){

        $this->browse(function (Browser $browser) {
            $browser->loginAs(1)
                ->visit('/admin/inhouse/product')
                ->assertSee('Product List');
        });
    }

    public function test_for_create_new_single_product(){

        $this->test_for_visit_index_page();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(2)')
                ->pause(1000)
                ->assertSee('Product Information')
                ->type('#product_name_new', $this->faker->name)
                ->type('#sku_single', $this->faker->slug)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(5) > div > input', $this->faker->name)
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(6) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(7) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(7) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(8) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(8) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(9) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(9) > div > div > ul > li:nth-child(2)')
                ->type('#minimum_order_qty', '2')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(11) > div > input', '10')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(1) > div.tagInput_field.mb_26 > div > input[type=text]', $this->faker->title)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(2) > div > div > div.note-editing-area > div.note-editable', $this->faker->paragraph)
                ->type('#selling_price', '250')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(5) > div > input', '10')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(6) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(7) > div > input', '5')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(8) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(8) > div > div > ul > li:nth-child(2)')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(10) > div > div > div.note-editing-area > div.note-editable', $this->faker->paragraph)
                ->type('#phisical_shipping_div > div > div > div > input', '0')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(16) > div > input', $this->faker->title)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(17) > div > textarea', $this->faker->paragraph)
                ->attach('#meta_image', __DIR__.'/files/mi.png')
                ->attach('#choice_form > div > div.col-lg-4 > div > div > div.col-lg-10 > div > div > button > input', __DIR__.'/files/product_details.png')
                ->attach('#choice_form > div > div.col-lg-4 > div > div > div:nth-child(5) > div > div > button > input', __DIR__.'/files/product_details_2.png')
                ->attach('#choice_form > div > div.col-lg-4 > div > div > div:nth-child(7) > div > div > button > input', __DIR__.'/files/digital_file.pdf')
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-lg-12.shipping_type_div > div > div')
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-lg-12.shipping_type_div > div > div > ul > li:nth-child(1)')
                ->type('#choice_form > div > div.col-lg-4 > div > div > div:nth-child(12) > div > input', $this->faker->slug)
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-12 > button')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',30)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });
    }

    public function test_for_create_digital_new_single_product(){
        $this->test_for_visit_index_page();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(2)')
                ->pause(1000)
                ->assertSee('Product Information')
                ->type('#product_name_new', $this->faker->name)
                ->type('#sku_single', $this->faker->slug)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(5) > div > input', $this->faker->name)
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(6) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(7) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(7) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(8) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(8) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(9) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(9) > div > div > ul > li:nth-child(2)')
                ->type('#minimum_order_qty', '2')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(11) > div > input', '10')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(1) > div.tagInput_field.mb_26 > div > input[type=text]', $this->faker->title)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(2) > div > div > div.note-editing-area > div.note-editable', $this->faker->paragraph)
                ->type('#selling_price', '250')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(5) > div > input', '10')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(6) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(7) > div > input', '5')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(8) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(8) > div > div > ul > li:nth-child(2)')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(10) > div > div > div.note-editing-area > div.note-editable', $this->faker->paragraph)
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-xl-12 > div > ul > li > label > span')
                ->pause(1000)
                ->attach('#digital_file', __DIR__.'/files/digital_file.pdf')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(16) > div > input', $this->faker->title)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(17) > div > textarea', $this->faker->paragraph)
                ->attach('#meta_image', __DIR__.'/files/mi.png')
                ->attach('#thumbnail_image', __DIR__.'/files/product_details.png')
                ->attach('#galary_image', __DIR__.'/files/product_details_2.png')
                ->attach('#pdf', __DIR__.'/files/digital_file.pdf')
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-12 > button')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });        
    }

    public function test_for_create__new_variant_product(){
        $this->test_for_visit_index_page();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(2)')
                ->pause(1000)
                ->assertSee('Product Information')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(2) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->type('#product_name_new', $this->faker->name)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(5) > div > input', $this->faker->name)
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(6) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(7) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(7) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(8) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(8) > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(9) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(9) > div > div > ul > li:nth-child(2)')
                ->click('#minimum_order_qty', '2')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div:nth-child(11) > div > input', '10')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div.col-lg-12.attribute_div > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(1) > div.col-lg-12.attribute_div > div > div > ul > li:nth-child(2)')
                ->waitFor('#customer_choice_options > div > div.col-lg-4 > div > input', 25)
                ->click('#customer_choice_options > div > div.col-lg-8 > div > div')
                ->click('#customer_choice_options > div > div.col-lg-8 > div > div > ul > li:nth-child(1)')
                ->click('#customer_choice_options > div > div.col-lg-8 > div > div')
                ->click('#customer_choice_options > div > div.col-lg-8 > div > div > ul > li:nth-child(2)')
                ->waitFor('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(1) > td.text-center.pt-36 > label', 30)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(1) > div.tagInput_field.mb_26 > div > input[type=text]', $this->faker->name)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(2) > div > div > div.note-editing-area > div.note-editable', $this->faker->paragraph)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(5) > div > input', '10')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(6) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(7) > div > input', '5')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(8) > div > div')
                ->click('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(8) > div > div > ul > li:nth-child(2)')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(10) > div > div > div.note-editing-area > div.note-editable', $this->faker->paragraph)
                ->type('#phisical_shipping_div > div > div > div > input', '0')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(1) > td:nth-child(2) > input', '200')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(1) > td:nth-child(3) > input', $this->faker->slug)
                ->attach('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(1) > td.variant_physical_div.text-center.pt_2 > div > div > button > input', __DIR__.'/files/product_details.png')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(2) > td:nth-child(2) > input', '210')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(2) > td:nth-child(3) > input', $this->faker->name)
                ->attach('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div.col-lg-12.sku_combination > table > tbody > tr:nth-child(2) > td.variant_physical_div.text-center.pt_2 > div > div > button > input', __DIR__.'/files/product_details_2.png')
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(16) > div > input', $this->faker->title)
                ->type('#choice_form > div > div.col-lg-8 > div > div:nth-child(3) > div:nth-child(17) > div > textarea', $this->faker->paragraph)
                ->attach('#meta_image', __DIR__.'/files/mi.png')
                ->attach('#thumbnail_image', __DIR__.'/files/product_details.png')
                ->attach('#galary_image', __DIR__.'/files/product_details_2.png')
                ->attach('#pdf', __DIR__.'/files/digital_file.pdf')
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-lg-12.shipping_type_div > div > div')
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-lg-12.shipping_type_div > div > div > ul > li:nth-child(2)')
                ->click('#choice_form > div > div.col-lg-4 > div > div > div.col-12 > button')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',30)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });        
    }


    public function test_for__with_stock_add_single_inhouse_product(){
        $this->test_for_create_new_single_product();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(3)')
                ->pause(1000)
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->pause(15000)
                ->click('#single_product_stock_manage_div > div > div')
                ->click('#single_product_stock_manage_div > div > div > ul > li:nth-child(1)')
                ->pause(1000)
                ->type('#product_stock', '100')
                ->type('#selling_prices', '220')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(1) > div > input', $this->faker->name)
                ->attach('#thumbnail_image_seller', __DIR__.'/files/product_details_2.png')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(3) > div > input', '10')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(5) > div > input', '8')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(5)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(5)')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });
    }


    public function test_for_without_stock_add_single_inhouse_product(){
        $this->test_for_create_new_single_product();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(3)')
                ->pause(1000)
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->pause(15000)

                ->type('#selling_prices', '220')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(1) > div > input', $this->faker->name)
                ->attach('#thumbnail_image_seller', __DIR__.'/files/product_details_2.png')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(3) > div > input', '10')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(5) > div > input', '8')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(5)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(5)')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });
    }

    public function test_for_with_stock_variant_product_add(){
        $this->test_for_create__new_variant_product();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(3)')
                ->pause(1000)
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->pause(15000)
                ->click('#single_product_stock_manage_div > div > div')
                ->click('#single_product_stock_manage_div > div > div > ul > li:nth-child(1)')
                ->click('#variant_sku_div > div > div')
                ->click('#variant_sku_div > div > div > ul > li:nth-child(1)')
                ->click('#variant_sku_div > div > div')
                ->click('#variant_sku_div > div > div > ul > li:nth-child(2)')
                ->waitFor('#sku_tbody > tr > td.text-center.product_sku_name', 30)
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(1) > div > input', $this->faker->name)
                ->attach('#thumbnail_image_seller', __DIR__.'/files/product_details_2.png')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(3) > div > input', '20')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div > ul > li:nth-child(1)')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(5) > div > input', '15')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div > ul > li:nth-child(1)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(5)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(4) > td:nth-child(4)')
                ->type('#sku_tbody > tr:nth-child(1) > td.text-center.sku_price_td > input', '210')
                ->type('#sku_tbody > tr:nth-child(1) > td.sku_price_td.stock_td > input', '50')
                ->type('#sku_tbody > tr:nth-child(2) > td.text-center.sku_price_td > input', '220')
                ->type('#sku_tbody > tr:nth-child(2) > td.sku_price_td.stock_td > input', '60')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });        
    }

    public function test_for_without_stock_variant_product_add(){
        $this->test_for_create__new_variant_product();
        $this->browse(function (Browser $browser) {
            $browser->click('#main-content > section > div > div.row > div.col-md-12.mb-20 > div > div > ul > li:nth-child(5) > a')
                ->assertPathIs('/admin/inhouse/products/create')
                ->assertSee('Add Product')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > div:nth-child(1) > div > div > div > ul > li:nth-child(3)')
                ->pause(1000)
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(2) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->pause(15000)
                ->click('#single_product_stock_manage_div > div > div')
                ->click('#single_product_stock_manage_div > div > div > ul > li:nth-child(1)')
                ->click('#variant_sku_div > div > div')
                ->click('#variant_sku_div > div > div > ul > li:nth-child(1)')
                ->click('#variant_sku_div > div > div')
                ->click('#variant_sku_div > div > div > ul > li:nth-child(2)')
                ->waitFor('#sku_tbody > tr > td.text-center.product_sku_name', 25)
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(1) > div > input', $this->faker->name)
                ->attach('#thumbnail_image_seller', __DIR__.'/files/product_details_2.png')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(3) > div > input', '20')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(4) > div > div > ul > li:nth-child(1)')
                ->type('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(5) > div > input', '15')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div')
                ->click('#exsisitng_product_div > form > div:nth-child(4) > div:nth-child(6) > div > div > ul > li:nth-child(1)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(5)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(4) > td:nth-child(4)')
                ->type('#sku_tbody > tr:nth-child(1) > td.text-center.sku_price_td > input', '210')
                ->type('#sku_tbody > tr:nth-child(2) > td.text-center.sku_price_td > input', '220')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product added Successfully!');
        });        
    }


    public function test_for_with_stock_edit_inhouse_single_product(){
        $this->test_for__with_stock_add_single_inhouse_product();
        $this->browse(function (Browser $browser) {
            $product = SellerProduct::where('user_id', auth()->user()->id)->latest()->first();
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a.dropdown-item.edit_brand')
                ->assertPathIs('/admin/inhouse/product/edit/'.$product->id)
                ->assertSee('Product Update')
                ->type('#product_stock', '50')
                ->type('#selling_price', '220')
                ->type('#tax', '15')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(2) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->type('#discount', '5')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(4) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(6)')
                ->click('#endDate')
                ->pause(1000)
                ->click('body > div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(3)')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product updated Successfully!');
        });
    }

    
    public function test_for_without_stock_edit_inhouse_single_product(){
        $this->test_for_without_stock_add_single_inhouse_product();
        $this->browse(function (Browser $browser) {
            $product = SellerProduct::where('user_id', auth()->user()->id)->latest()->first();
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a.dropdown-item.edit_brand')
                ->assertPathIs('/admin/inhouse/product/edit/'.$product->id)
                ->assertSee('Product Update')
                ->type('#selling_price', '220')
                ->type('#tax', '15')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(2) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->type('#discount', '5')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(4) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(6)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(3)')
                ->pause(1000)
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product updated Successfully!');
        });
    }

    public function test_for_stock_to_not_stock_edit_inhouse_single_product(){
        $this->test_for_without_stock_add_single_inhouse_product();
        $this->browse(function (Browser $browser) {
            $product = SellerProduct::where('user_id', auth()->user()->id)->latest()->first();
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a.dropdown-item.edit_brand')
                ->assertPathIs('/admin/inhouse/product/edit/'.$product->id)
                ->assertSee('Product Update')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(3) > div > div > label.switch_toggle > div')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Updated Successfully.')
                ->assertPathIs('/admin/inhouse/product/edit/'.$product->id)
                ->type('#product_stock', '50')
                ->type('#selling_price', '220')
                ->type('#tax', '15')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(2) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->type('#discount', '5')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(4) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(6)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-bottom > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(3)')
                ->pause(1000)
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product updated Successfully!');
        });
    }

    public function test_for_with_stock_edit_inhouse_variant_product(){
        $this->test_for_with_stock_variant_product_add();
        $this->browse(function (Browser $browser) {
            $product = SellerProduct::where('user_id', auth()->user()->id)->latest()->first();
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a.dropdown-item.edit_brand')
                ->assertPathIs('/admin/inhouse/product/edit/'.$product->id)
                ->assertSee('Product Update')
                ->type('#tax', '11')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(2) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->type('#discount', '7')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(4) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(6)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(5)')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div > ul > li:nth-child(2)')
                ->waitFor('#sku_tbody > tr:nth-child(3) > td:nth-child(2)', 25)
                ->click('#sku_tbody > tr:nth-child(3) > td.text-center.sku_delete_td > p > i')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div > ul > li:nth-child(2)')
                ->type('#sku_tbody > tr:nth-child(1) > td:nth-child(3) > input', '221')
                ->type('#sku_tbody > tr:nth-child(1) > td:nth-child(4) > input', '110')
                ->type('#sku_tbody > tr:nth-child(2) > td:nth-child(3) > input', '240')
                ->type('#sku_tbody > tr:nth-child(2) > td:nth-child(4) > input', '50')
                ->type('#sku_tbody > tr:nth-child(3) > td.text-center.sku_price_td > input','250')
                ->type('#sku_tbody > tr:nth-child(3) > td:nth-child(4) > input', '30')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',30)
                ->assertSeeIn('.toast-message', 'Product updated Successfully!');
        });        
    }

    public function test_for_without_stock_edit_inhouse_variant_product(){
        $this->test_for_without_stock_variant_product_add();
        $this->browse(function (Browser $browser) {
            $product = SellerProduct::where('user_id', auth()->user()->id)->latest()->first();
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a.dropdown-item.edit_brand')
                ->assertPathIs('/admin/inhouse/product/edit/'.$product->id)
                ->assertSee('Product Update')
                ->type('#tax', '11')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(2) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(2) > div > div > ul > li:nth-child(2)')
                ->type('#discount', '7')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(4) > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(4) > div:nth-child(4) > div > div > ul > li:nth-child(2)')
                ->click('#startDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(1) > td:nth-child(6)')
                ->click('#endDate')
                ->pause(1000)
                ->click('div.datepicker.datepicker-dropdown.dropdown-menu.datepicker-orient-left.datepicker-orient-top > div.datepicker-days > table > tbody > tr:nth-child(5) > td:nth-child(5)')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div > ul > li:nth-child(2)')
                ->waitFor('#sku_tbody > tr:nth-child(3) > td:nth-child(2)', 25)
                ->click('#sku_tbody > tr:nth-child(3) > td.text-center.sku_delete_td > p > i')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div')
                ->click('#main-content > section > div > div > div:nth-child(2) > div > form > div:nth-child(5) > div.col-xl-6 > div > div > ul > li:nth-child(2)')
                ->type('#sku_tbody > tr:nth-child(1) > td:nth-child(3) > input', '221')
                ->type('#sku_tbody > tr:nth-child(2) > td:nth-child(3) > input', '240')
                ->type('#sku_tbody > tr:nth-child(3) > td.text-center.sku_price_td > input','250')
                ->click('#save_button_parent')
                ->assertPathIs('/admin/inhouse/product')
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product updated Successfully!');
        });        
    }

    public function test_delete_inhouse_product(){
        $this->test_for_without_stock_add_single_inhouse_product();
        $this->browse(function (Browser $browser) {
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a:nth-child(3)')
                ->whenAvailable('#item_delete_form', function($modal){
                    $modal->click('input.primary-btn.fix-gr-bg');

                })
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Product Deleted successfully!');
        });
    }

    public function test_show_inhouse_product(){
        $this->test_for_without_stock_add_single_inhouse_product();
        $this->browse(function (Browser $browser) {
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > button')
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(8) > div > div > a:nth-child(1)')
                ->whenAvailable('#productDetails > div', function($modal){
                    $modal->assertSeeAnythingIn('div > div.modal-header > h4');

                });
                
        });
    }

    public function test_for_change_inhouse_product_status(){
        $this->test_for_without_stock_add_single_inhouse_product();
        $this->browse(function (Browser $browser) {
            $browser->waitFor('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(7) > label > div', 30)
                ->click('#sellerProductTable > tbody > tr:nth-child(1) > td:nth-child(7) > label > div')
                ->pause(8000)
                ->waitFor('.toast-message',25)
                ->assertSeeIn('.toast-message', 'Updated Successfully.');
                
        });
    }

}
