<?php

namespace App\Http\Controllers;

use App\Mail\InvoiceEmailManager;
use App\Models\Address;
use App\Models\City;
use App\Models\Country;
use App\Models\DeliveryBoy;
use App\Models\DeliveryBoyCollection;
use App\Models\DeliveryBoyPayment;
use App\Models\DeliveryHistory;
use App\Models\Order;
use App\Models\State;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;

class DeliveryBoyController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:view_all_delivery_boy'])->only('index');
        $this->middleware(['permission:delivery_boy_configuration'])->only('configuration');
    }


    /**
     * Configure the delivery boy 
     *
     * @param  \App\Models\DeliveryBoy  $deliveryBoy
     * @return \Illuminate\Http\Response
     */
    public function configuration()
    {
        return view('backend.delivery_boy.configuration');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $delivery_boys = User::with('delivery_boy')->where('user_type', 'delivery_boy')->orderBy('created_at', 'desc');
        if ($request->has('search')) {
            $sort_search = $request->search;
            $delivery_boys = $delivery_boys->where('name', 'like', '%' . $sort_search . '%')->orWhere('email', 'like', '%' . $sort_search . '%');
        }
        $delivery_boys = $delivery_boys->paginate(15);
        return view('backend.delivery_boy.index', compact('delivery_boys'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $countries = Country::where('status', 1)->get();
        return view('backend.delivery_boy.create', compact('countries'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name'          => 'required',
            'email'         => 'required|unique:users|max:255',
            'phone'         => 'required|unique:users|max:15',
            'password'      => 'required|min:6',
            'country_id'    => 'required',
            'state_id'      => 'required',
            'city_id'       => 'required',
        ]);

        $country = Country::where('id', $request->country_id)->first();
        $state = State::where('id', $request->state_id)->first();
        $city = City::where('id', $request->city_id)->first();

        $user                       = new User;
        $user->user_type            = 'delivery_boy';
        $user->name                 = $request->name;
        $user->email                = $request->email;
        $user->phone                = $request->phone;
        $user->avatar               = $request->avatar_original;
        $user->email_verified_at    = date("Y-m-d H:i:s");
        $user->password             = Hash::make($request->password);
        $user->save();

        $address = new Address;
        $address->user_id = $user->id;
        $address->country = $country->name;
        $address->country_id = $country->id;
        $address->state = $state->name;
        $address->state_id = $state->id;
        $address->city = $city->name;
        $address->city_id = $city->id;
        $address->address = $request->address;
        $address->save();

        $delivery_boy = new DeliveryBoy;

        $delivery_boy->user_id = $user->id;
        $delivery_boy->save();

        flash(translate('Delivery Boy has been created successfully'))->success();
        return redirect()->route('delivery-boy.index');
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\DeliveryBoy  $deliveryBoy
     * @return \Illuminate\Http\Response
     */
    public function show(DeliveryBoy $deliveryBoy)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\DeliveryBoy  $deliveryBoy
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $countries = Country::where('status', 1)->get();
        $states = State::where('status', 1)->get();
        $cities = City::where('status', 1)->get();
        $delivery_boy = User::findOrFail($id);
        $address = Address::where('user_id', $delivery_boy->id)->first();
        return view('backend.delivery_boy.edit', compact('delivery_boy', 'countries', 'states', 'cities', 'address'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\DeliveryBoy  $deliveryBoy
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request,  $id)
    {
        $delivery_boy = User::find($id);

        $request->validate([
            'name'       => 'required',
            'email'      => 'required|unique:users,email,' . $delivery_boy->id,
            'phone'      => 'required',
            'country_id' => 'required',
            'state_id'   => 'required',
            'city_id'    => 'required',
        ]);

        $country = Country::where('id', $request->country_id)->first();
        $state = State::where('id', $request->state_id)->first();
        $city = City::where('id', $request->city_id)->first();

        $delivery_boy->name             = $request->name;
        $delivery_boy->email            = $request->email;
        $delivery_boy->phone            = $request->phone;
        $delivery_boy->avatar  = $request->avatar_original;

        if (strlen($request->password) > 0) {
            $delivery_boy->password = Hash::make($request->password);
        }
        $delivery_boy->save();

        $address = $delivery_boy->addresses->first();
        $address->country = $country->name;
        $address->country_id = $country->id;
        $address->state = $state->name;
        $address->state_id = $state->id;
        $address->city = $city->name;
        $address->city_id = $city->id;
        $address->address = $request->address;
        $address->save();

        flash(translate('Delivery Boy has been updated successfully'))->success();
        return back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\DeliveryBoy  $deliveryBoy
     * @return \Illuminate\Http\Response
     */
    public function destroy(DeliveryBoy $deliveryBoy)
    {
        //
    }

    public function assign_delivery_boy(Request $request)
    {
        if (get_setting('delivery_boy') == 1) {

            $order = Order::findOrFail($request->order_id);
            $order->assign_delivery_boy = $request->delivery_boy;
            $order->delivery_history_date = date("Y-m-d H:i:s");
            $order->save();

            $delivery_history = DeliveryHistory::where('order_id', $order->id)
                ->where('delivery_status', $order->delivery_status)
                ->first();

            if (empty($delivery_history)) {
                $delivery_history = new DeliveryHistory;

                $delivery_history->order_id = $order->id;
                $delivery_history->delivery_status = $order->delivery_status;
                $delivery_history->payment_type = $order->payment_type;
            }
            $delivery_history->delivery_boy_id = $request->delivery_boy;

            $delivery_history->save();

            if (env('MAIL_USERNAME') != null) {
                $array['view'] = 'emails.invoice';
                $array['subject'] = translate('You are assigned to delivery an order. Order code') . ' - ' . $order->code;
                $array['from'] = env('MAIL_FROM_ADDRESS');
                $array['order'] = $order;

                try {
                    Mail::to($order->delivery_boy->email)->queue(new InvoiceEmailManager($array));
                } catch (\Exception $e) {
                }
            }
        }

        return 1;
    }

    public function store_delivery_history($order)
    {
        $user = auth()->user();
        $delivery_history = new DeliveryHistory;

        $delivery_history->order_id         = $order->id;
        $delivery_history->delivery_boy_id  = $user->id;
        $delivery_history->delivery_status  = $order->delivery_status;
        $delivery_history->payment_type     = $order->payment_type;
        if ($order->delivery_status == 'delivered') {
            $delivery_boy = DeliveryBoy::where('user_id', $user->id)->first();

            $delivery_history->earning      = get_setting('delivery_boy_commission');
            $delivery_boy->total_earning   += get_setting('delivery_boy_commission');
            if ($order->payment_type == 'cash_on_delivery') {
                $delivery_history->collection    = $order->grand_total;
                $delivery_boy->total_collection += $order->grand_total;
                $order->payment_status           = 'paid';
            }
            $delivery_boy->save();
        }
        $order->delivery_history_date = date("Y-m-d H:i:s");
        $order->save();
        $delivery_history->save();
    }

    public function cancel_request_list()
    {
        $user = auth()->user();
        $order_query = Order::query();
        if ($user->user_type == 'delivery_boy') {
            $order_query = $order_query->with('combined_order')->where('assign_delivery_boy', $user->id);
        }
        $order_query = $order_query->where('delivery_status', '!=', 'cancelled')->where('cancel_request', 1);
        $cancel_requests = $order_query->paginate(10);

        return view('backend.delivery_boy.cancel_request_list', compact('cancel_requests'));
    }

    public function order_detail($id)
    {
        $order = Order::findOrFail(decrypt($id));
        return view('delivery_boy.order_detail', compact('order'));
    }
    public function payment_histories()
    {

        $delivery_boy_payment_query = DeliveryBoyPayment::query();
        if (auth()->user()->user_type == 'delivery_boy') {
            $delivery_boy_payment_query = $delivery_boy_payment_query->where('user_id', auth()->user()->id);
        }
        $delivery_boy_payments = $delivery_boy_payment_query->paginate(10);

        return view('backend.delivery_boy.delivery_boys_payment_list', compact('delivery_boy_payments'));
    }
    public function collection_histories()
    {

        $delivery_boy_collection_query = DeliveryBoyCollection::query();
        if (auth()->user()->user_type == 'delivery_boy') {
            $delivery_boy_collection_query = $delivery_boy_collection_query->where('user_id', auth()->user()->id);
        }
        $delivery_boy_collections = $delivery_boy_collection_query->paginate(10);

        return view('backend.delivery_boy.delivery_boys_collection_list', compact('delivery_boy_collections'));
    }
    public function deliveryboycollection()
    {
    }
    public function order_collection_form(Request $request)
    {

        $delivery_boy_info = DeliveryBoy::with('user')
            ->where('user_id', $request->id)
            ->first();

        return view('backend.delivery_boy.order_collection_form', compact('delivery_boy_info'));
    }
    public function delivery_earning_form(Request $request)
    {

        $delivery_boy_info = DeliveryBoy::with('user')
            ->where('user_id', $request->id)
            ->first();

        return view('backend.delivery_boy.delivery_earning_form', compact('delivery_boy_info'));
    }
    public function collection_from_delivery_boy(Request $request)
    {

        $delivery_boy = DeliveryBoy::where('user_id', $request->delivery_boy_id)->first();

        if ($request->payout_amount > $delivery_boy->total_collection) {
            flash(translate('Collection Amount Can Not Be Larger Than Collected Amount'))->error();
            return redirect()->back();
        }

        $delivery_boy->total_collection -= $request->payout_amount;

        if ($delivery_boy->save()) {
            $delivery_boy_collection          = new DeliveryBoyCollection;
            $delivery_boy_collection->user_id = $request->delivery_boy_id;
            $delivery_boy_collection->collection_amount = $request->payout_amount;

            $delivery_boy_collection->save();

            flash(translate('Collection From Delivery Boy Successfully'))->success();
        } else {
            flash(translate('Something went wrong'))->error();
        }

        return redirect()->route('delivery-boy.index');
    }
    public function paid_to_delivery_boy(Request $request)
    {

        $delivery_boy = DeliveryBoy::where('user_id', $request->delivery_boy_id)->first();

        if ($request->paid_amount > $delivery_boy->total_earning) {
            flash(translate('Paid Amount Can Not Be Larger Than Payable Amount'))->error();
            return redirect()->back();
        }

        $delivery_boy->total_earning -= $request->paid_amount;

        if ($delivery_boy->save()) {
            $delivery_boy_payment          = new DeliveryBoyPayment;
            $delivery_boy_payment->user_id = $request->delivery_boy_id;
            $delivery_boy_payment->payment = $request->paid_amount;

            $delivery_boy_payment->save();

            flash(translate('Pay To Delivery Boy Successfully'))->success();
        } else {
            flash(translate('Something went wrong'))->error();
        }

        return redirect()->route('delivery-boy.index');
    }

    public function delivery_boy_ban($id)
    {
        $delivery_boy = User::find($id);

        if ($delivery_boy->banned == 1) {
            $delivery_boy->banned = 0;
            flash(translate('Delivery Boy UnBanned Successfully'))->success();
        } else {
            $delivery_boy->banned = 1;
            flash(translate('Delivery Boy Banned Successfully'))->success();
        }

        $delivery_boy->save();

        return back();
    }
}
